const express = require('express');
const cors = require('cors');
const mysql = require('mysql2/promise');
const path = require('path');
require('dotenv').config();

const app = express();
const PORT = process.env.PORT || 3001;

// Middleware
app.use(cors());
app.use(express.json());

// Database Connection Pool
const pool = mysql.createPool({
    host: process.env.DB_HOST || 'localhost',
    user: process.env.DB_USER || 'root',
    password: process.env.DB_PASSWORD || '',
    database: process.env.DB_NAME || 'rise_eds_db',
    waitForConnections: true,
    connectionLimit: 10,
    queueLimit: 0
});

// Test DB Connection
pool.getConnection()
    .then(conn => {
        console.log("Connected to MySQL Database.");
        conn.release();
    })
    .catch(err => {
        console.error("Database Connection Failed:", err);
    });

// API Routes
app.post('/api/license/register', async (req, res) => {
    const { email, licenseKey } = req.body;
    try {
        // Check if table exists, if not create strings (Quick migration for demo)
        const [rows] = await pool.query("SHOW TABLES LIKE 'licenses'");
        if (rows.length === 0) {
            await pool.query(`
        CREATE TABLE licenses (
          id INT AUTO_INCREMENT PRIMARY KEY,
          email VARCHAR(255) NOT NULL,
          license_key VARCHAR(255) NOT NULL,
          created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
      `);
        }

        await pool.query('INSERT INTO licenses (email, license_key) VALUES (?, ?)', [email, licenseKey]);
        res.json({ success: true, message: 'License registered successfully' });
    } catch (err) {
        console.error(err);
        res.status(500).json({ error: 'Database error' });
    }
});

app.get('/api/licenses', async (req, res) => {
    try {
        // Check if table exists
        const [rows] = await pool.query("SHOW TABLES LIKE 'licenses'");
        if (rows.length === 0) return res.json([]);

        const [results] = await pool.query('SELECT * FROM licenses');
        // Map to AdminProfile shape for frontend compatibility
        const admins = results.map((row, idx) => ({
            id: `ADMIN_${row.id}`,
            firstName: 'Admin',
            lastName: 'User',
            email: row.email,
            role: 'Admin',
            permissions: ['super_admin'],
            accessKey: row.license_key,
            status: 'Active'
        }));
        res.json(admins);
    } catch (err) {
        console.error(err);
        res.status(500).json({ error: 'Database error' });
    }
});

// Serve React App in Production
if (process.env.NODE_ENV === 'production') {
    app.use(express.static(path.join(__dirname, 'dist')));
    app.get('*', (req, res) => {
        res.sendFile(path.join(__dirname, 'dist', 'index.html'));
    });
}

app.listen(PORT, () => {
    console.log(`Server running on port ${PORT}`);
});
